package algs42;
import stdlib.*;
import algs35.ST;
/* ***********************************************************************
 *  Compilation:  javac SymbolDigraph.java
 *  Execution:    java SymbolDigraph
 *  Dependencies: ST.java Digraph.java In.java
 *
 *  %  java SymbolDigraph routes.txt " "
 *  JFK
 *     MCO
 *     ATL
 *     ORD
 *  ATL
 *     HOU
 *     MCO
 *  LAX
 *
 *************************************************************************/

public class SymbolDigraph {
	private final ST<String, Integer> st;  // string -> index
	private final String[] keys;           // index  -> string
	private final Digraph G;
	private final String delimiter;

	public SymbolDigraph(String filename, String delimiter) {
		this.delimiter = delimiter;
		this.st = new ST<>();

		// First pass builds the index by reading strings to associate
		// distinct strings with an index
		In in = new In(filename);
		while (in.hasNextLine()) {
			String[] a = in.readLine().split(delimiter);
			for (int i = 0; i < a.length; i++) {
				if (!st.contains(a[i]))
					st.put(a[i], st.size());
			}
		}

		// inverted index to get string keys in an aray
		this.keys = new String[st.size()];
		for (String name : st.keys()) {
			keys[st.get(name)] = name;
		}

		// second pass builds the digraph by connecting first vertex on each
		// line to all others
		this.G = new Digraph(st.size());
		in = new In(filename);
		while (in.hasNextLine()) {
			String[] a = in.readLine().split(delimiter);
			int v = st.get(a[0]);
			for (int i = 1; i < a.length; i++) {
				int w = st.get(a[i]);
				G.addEdge(v, w);
			}
		}
	}

	public boolean contains(String s) {
		return st.contains(s);
	}

	public int index(String s) {
		return st.get(s);
	}

	public String name(int v) {
		return keys[v];
	}

	public Digraph G() {
		return G;
	}

	public String toString() {
		StringBuilder s = new StringBuilder();
		String NEWLINE = System.getProperty("line.separator");
		s.append(G.V() + " vertices, " + G.E() + " edges " + NEWLINE);
		for (int v = 0; v < G.V(); v++) {
			s.append(name(v));
			for (int w : G.adj (v)) {
				s.append(delimiter);
				s.append(name(w));
			}
			s.append(NEWLINE);
		}
		return s.toString();
	}

	public static void main(String[] args) {
		args = new String[] { "data/jobs.txt", "/" };

		String filename  = args[0];
		String delimiter = args[1];
		SymbolDigraph sg = new SymbolDigraph(filename, delimiter);
		StdOut.println (sg);

		Digraph G = sg.G();
		while (!StdIn.isEmpty()) {
			String t = StdIn.readLine();
			for (int v : G.adj(sg.index(t))) {
				StdOut.println("   " + sg.name(v));
			}
		}
	}
}
